//
//  GeometryGamesExportView.swift
//  Crystal Flight
//
//  Created by Jeff on 12/12/20.
//

import SwiftUI


struct GeometryGamesExportView: View {

	//	For reasons I don't understand (am still new
	//	to SwiftUI in general, and Pickers in particular),
	//	the Picker was unresponsive when given a UInt
	//	for the selection variable, but it works fine
	//	now that the selection variable is this enum.
	enum ExportedImageSize: UInt, CaseIterable, Identifiable {

		case size256  =  256
		case size512  =  512
		case size1024 = 1024
		case size2048 = 2048
		case size4096 = 4096
		
		var id: UInt { self.rawValue }
		
		var asString: String {
			switch self {
			case .size256:
				return "256 × 256"
			case .size512:
				return "512 × 512"
			case .size1024:
				return "1024 × 1024"
			case .size2048:
				return "2048 × 2048"
			case .size4096:
				return "4096 × 4096"
			}
		}
	}

	@State var imageSizeInPixels = ExportedImageSize.size256	//	to be overridden in onAppear below
	let theDefaultExportedImageSizeAsUInt: UInt = 1024
	let theDefaultExportedImageSizeAsEnum = ExportedImageSize.size1024

	let geometryGamesView: GeometryGamesView
	
	
	var body: some View {
	
		let theButtonPadding: CGFloat = 4
		
		let theImageSizeString = Bundle.main.localizedString(
			forKey: "Image Size",
			value: nil,
			table: "GeometryGamesLocalizable")

		let theSaveImageString = Bundle.main.localizedString(
			forKey: "Save Image",
			value: nil,
			table: "GeometryGamesLocalizable")

		let theCopyImageString = Bundle.main.localizedString(
			forKey: "Copy Image",
			value: nil,
			table: "GeometryGamesLocalizable")

		return VStack(alignment: .center) {
			
			HStack() {

//	The Picker worked fine when running on an iOS device,
//	but didn't work at all when running on a Mac.
//
//				Picker(
//					theImageSizeString,
//					selection: $imageSizeInPixels
//				) {
//					ForEach(ExportedImageSize.allCases, id: \.id) {theImageSize in
//						Text(verbatim: theImageSize.asString).tag(theImageSize)
//					}
//				}
//				.pickerStyle(MenuPickerStyle())
//
//	So instead let's let a Menu contain the Picker, which should be just as good.
//	Well, on macOS it inserts an extra level in the menu heirarchy, that is,
//
//		Image Size > Image Size > {options}
//
//	instead of just
//
//		Image Size > {options}
//
//	as on iOS, but I guess we can live with that.
//
				Menu() {
					Picker(selection: $imageSizeInPixels, label: Text(theImageSizeString)) {
						ForEach(ExportedImageSize.allCases, id: \.id) {theImageSize in
							Text(theImageSize.asString)
								.tag(theImageSize)
						}
					}
				} label: {
					Text(theImageSizeString)
				}

				Spacer()

				Text(formattedImageSize(imageSizeInPixels.rawValue))
					.foregroundColor(.gray)
			}
			
			//	Caution:  Divider() will make the HStack fill all available width
			//	unless we apply .fixedSize() or .fixedSize(horizontal: vertical: )
			//	to the containing VStack().
			Divider()
		
			Button() {
		
				let (_, _) = geometryGamesView.saveImage(
					requestedWidthPx:  imageSizeInPixels.rawValue,
					requestedHeightPx: imageSizeInPixels.rawValue)
				
			} label: {
			
				HStack() {
					Text(theSaveImageString)
					Spacer()
					Image(systemName: "folder")
				}
				.padding(EdgeInsets(top: theButtonPadding, leading: 0, bottom: theButtonPadding, trailing: 0))
			}

			Button {
		
				let (_, _) = geometryGamesView.copyImage(
					requestedWidthPx:  imageSizeInPixels.rawValue,
					requestedHeightPx: imageSizeInPixels.rawValue)

			} label: {
			
				HStack() {
					Text(theCopyImageString)
					Spacer()
					Image(systemName: "doc.on.doc")
				}
				.padding(EdgeInsets(top: theButtonPadding, leading: 0, bottom: theButtonPadding, trailing: 0))
			}
		}
		.fixedSize()
		.onAppear() {
			imageSizeInPixels = ExportedImageSize(rawValue:
				UserDefaults.standard.object(forKey: "exported image size")
					as? UInt ?? theDefaultExportedImageSizeAsUInt
			) ?? theDefaultExportedImageSizeAsEnum
		}
		.onDisappear() {
			UserDefaults.standard.setValue(
				imageSizeInPixels.rawValue,
				forKey: "exported image size")
		}
		.padding(geometryGamesPanelPadding)
		.background(Color.white)
		.cornerRadius(geometryGamesCornerRadius)
	}
	
	func formattedImageSize(_ size: UInt) -> String {
		
		let theNumberFormatter = NumberFormatter()
		theNumberFormatter.groupingSeparator = ""	//	suppress the comma in "1,024" etc.

		let theFormattedNumber = (theNumberFormatter.string(from: NSNumber(value: size))
									?? "error")
		
		let theDimensions = theFormattedNumber + " × " + theFormattedNumber
		
		return theDimensions
	}
}
